/**
  ******************************************************************************
  * @file    main.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   Program entry point
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 
#define MUTEXASOBJECT
/* Includes ------------------------------------------------------------------*/
extern "C"
{
    #include "stm32f4_discovery.h"
    #include "freertos.h"
    #include "task.h"
#ifdef MUTEXASOBJECT
    #include "semphr.h" 
#endif
}
  
#define LED_ORANGE LED3
#define LED_GREEN  LED4
#define LED_BLUE   LED6
#define LED_RED    LED5

#define BLINKEDELAYINMS 1000
#define BLINKTASKPRIORITY (configMAX_PRIORITIES-2)

#ifdef MUTEXASOBJECT
class Exclusive_Access
{
    private:
        xSemaphoreHandle   m_SemId;
    public:
        Exclusive_Access(xSemaphoreHandle p_SemId)
        {
            xSemaphoreTakeRecursive(p_SemId,portMAX_DELAY);
            m_SemId = p_SemId;
        }
        ~Exclusive_Access(void)
        {
            xSemaphoreGiveRecursive(m_SemId);
        }
};

#endif

/** @brief Callback funtion to handle programming errors
 *
 *  Handles attempts to initialize instances of pure virtual classes (forbidden by definition). 
 *  @return none
 */

extern "C" {
  void __cxa_pure_virtual()
  {
    // put error handling here
  }
}

/** @brief override of the allocator
 *
 *
 *  @param p_Size size of memory block.
 *  @return pointer to block or 0
 */

void *operator new(size_t p_Size)
{
    return pvPortMalloc(p_Size);     
}

/** @brief override of the deallocator
 *
 *
 *  @param p_Object pointer to object being deallocated.
 *  @return none
 */

void operator delete (void *p_Object)
{
    vPortFree(p_Object);
}


void DispatcherFn(void *p_Param);  // forward declaration

/*! Base class for FreeRTOS task abstractions */

class FreeRTOSTask
{
  protected:
    virtual void TaskFn(void *) = 0;
    TaskHandle_t m_CreatedTask;
    void *m_Param;
  friend void DispatcherFn(void *p_Param);
  public:  
    FreeRTOSTask(const char * const p_Name,uint16_t p_StackDepth,void *const p_Param,UBaseType_t p_Priority)
    {
        m_Param = p_Param;
        xTaskCreate(DispatcherFn,p_Name,p_StackDepth,this,p_Priority,&m_CreatedTask);
    }
};

/*! derived class that implements a blinking task */

class BlinkTask : public FreeRTOSTask
{
  protected:
    void TaskFn(void *p_Param)
    {
      while (1)
      {
        STM_EVAL_LEDToggle(LED_ORANGE);
        STM_EVAL_LEDToggle(LED_BLUE);

        vTaskDelay(BLINKEDELAYINMS);

        STM_EVAL_LEDToggle(LED_GREEN);
        STM_EVAL_LEDToggle(LED_RED);

        vTaskDelay(BLINKEDELAYINMS);
      }  
    };
  public:
    BlinkTask(const char * const p_Name,uint16_t p_StackDepth,void *const p_Param,UBaseType_t p_Priority) : FreeRTOSTask(p_Name,p_StackDepth,p_Param,p_Priority) {};  
};

/** @brief Helper function to avoid using static C++ members
 *
 *
 *  @param p_Param Instance of a FreeRTOSTask derived object. 
 *  @return none. Function normally does not return.
 */

void DispatcherFn(void *p_Param)
{
    FreeRTOSTask *a_TaskObj = (FreeRTOSTask *)p_Param;
    (a_TaskObj->TaskFn(a_TaskObj->m_Param));
}

/** @brief Application entry point
 *
 *
 *  @return Dummy (required by unused C Runtime Code). Function does not return unless vTaskStartScheduler() has a problem.
 */

extern "C"
{
int main(void);
}

int main(void)
{  
  /* Initialize Leds mounted on STM32F4-Discovery board */
  STM_EVAL_LEDInit(LED_GREEN);
  STM_EVAL_LEDInit(LED_ORANGE);
  STM_EVAL_LEDInit(LED_RED);
  STM_EVAL_LEDInit(LED_BLUE);

  FreeRTOSTask *a_Task = new BlinkTask("BlinkTask", configMINIMAL_STACK_SIZE,0, BLINKTASKPRIORITY);
  /* Start the scheduler */
	vTaskStartScheduler();
}
